<?php

namespace App\Http\Controllers\Admin\Product;

use App\Http\Controllers\Controller;
use App\Http\Requests\Product\CategoryRequest;
use App\Models\Photo;
use App\Models\ProductCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use productbon\productbon;

class CategoryController extends Controller
{
    public function controller_title($type)
    {
        switch ($type) {
            case 'index':
                return 'دسته محصولات';
                break;
            case 'create':
                return 'افزودن دسته';
                break;
            case 'edit':
                return 'ویرایش دسته';
                break;
            case 'url_back':
                return route('admin.product-category.index');
                break;
            default:
                return '';
                break;
        }
    }

    public function __construct()
    {
        $this->middleware('permission:product_category_list', ['only' => ['index', 'show']]);
        $this->middleware('permission:product_category_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:product_category_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:product_category_delete', ['only' => ['destroy']]);
        $this->middleware('permission:product_category_sort', ['only' => ['sort']]);
    }

    public function index()
    {
        $items = ProductCategory::where('parent_id', null)->orderBy('sort')->get();
        return view('admin.product.category.index', compact('items'), ['title' => $this->controller_title('index')]);
    }

    public function show($id) {}

    public function create()
    {
        $url_back = $this->controller_title('url_back');
        $cats = ProductCategory::where('parent_id', null)->orderBy('sort')->get();
        return view('admin.product.category.create', compact('url_back', 'cats'), ['title' => $this->controller_title('create')]);
    }

    public function store(CategoryRequest $request)
    {
        try {
            $item = ProductCategory::create([
                'parent_id' => $request->parent_id != 'father' ? $request->parent_id : null,
                'title' => $request->title,
                'text' => $request->text,
                'status' => $request->status,
                'slug' => $request->slug
            ]);

            if ($request->hasFile('photo')) {
                $photo = new Photo();
                $photo->type = 'photo';
                $photo->path = file_store($request->photo, 'assets/admin/uploads/product-category' . '/');
                $item->photo()->save($photo);
            }

            store_lang($item, $request, ['title', 'text', 'status'], 'create');

            return redirect($this->controller_title('url_back'))->with('flash_message', 'اطلاعات با موفقیت افزوده شد');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', 'برای افزودن به مشکل خوردیم، مجدد تلاش کنید');
        }
    }

    public function edit($id)
    {
        $url_back = $this->controller_title('url_back');
        $item = ProductCategory::findOrFail($id);
        $cats = ProductCategory::where('parent_id', null)->where('id', '!=', $id)->orderBy('sort')->get();
        return view('admin.product.category.edit', compact('url_back', 'item', 'cats'), ['title' => $this->controller_title('edit')]);
    }

    public function update(CategoryRequest $request, $id)
    {
        $item = ProductCategory::findOrFail($id);
        try {
            ProductCategory::where('id', $id)->update([
                'parent_id' => $request->parent_id != 'father' ? $request->parent_id : null,
                'title' => $request->title,
                'text' => $request->text,
                'status' => $request->status,
                'slug' => $request->slug
            ]);
            //edit category
            if ($request->hasFile('photo')) {
                if ($item->photo) {
                    if (is_file($item->photo->path)) {
                        File::delete($item->photo->path);
                        delete_directory($item->photo->path);
                    }
                    $item->photo->delete();
                }
                $photo = new Photo();
                $photo->type = 'photo';
                $photo->path = file_store($request->photo, 'assets/admin/uploads/product-category' . '/');
                $item->photo()->save($photo);
            }

            store_lang($item, $request, ['title', 'text', 'status'], 'edit');

            return redirect($this->controller_title('url_back'))->with('flash_message', 'اطلاعات با موفقیت ویرایش شد');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', 'برای ویرایش به مشکل خوردیم، مجدد تلاش کنید');
        }
    }

    public function destroy($id)
    {
        $item = ProductCategory::findOrFail($id);
        try {
            $item->delete();
            return redirect($this->controller_title('url_back'))->with('flash_message', 'اطلاعات با موفقیت حذف شد');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', 'برای حذف به مشکل خوردیم، مجدد تلاش کنید');
        }
    }

    public function sort($id, Request $request)
    {
        $item = ProductCategory::findOrFail($id);
        try {
            $item->sort = $request->sort;
            $item->update();
            return redirect($this->controller_title('url_back'))->with('flash_message', 'اطلاعات با موفقیت سورت شد');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('err_message', 'برای سورت به مشکل خوردیم، مجدد تلاش کنید');
        }
    }
}
